# -*- coding: utf-8 -*-
# PiHoleLookup.py
# Autopsy Data Source Ingest Module
# Version 1.0 - Final Production (IP Logic Fix)
# Plugin Made from Lars Blomgaard in collaboration with ChatPGPT and GEMENI 3.0
# - Its a free plugin and you can use it wowever you like. 
# REMEMEBER to change the IP for your PI-hole. look for # <- CHANGE ME 


import re
import subprocess
from java.util.logging import Level
from java.util import ArrayList

from org.sleuthkit.autopsy.casemodule import Case
from org.sleuthkit.autopsy.ingest import IngestModuleFactoryAdapter
from org.sleuthkit.autopsy.ingest import DataSourceIngestModule
from org.sleuthkit.autopsy.ingest import IngestServices
from org.sleuthkit.autopsy.ingest import ModuleDataEvent
from org.sleuthkit.autopsy.ingest import IngestModule
from org.sleuthkit.datamodel import BlackboardArtifact
from org.sleuthkit.datamodel import BlackboardAttribute

class PiHoleLookupIngestModuleFactory(IngestModuleFactoryAdapter):

    moduleName = "Pi-hole URL Check"

    def getModuleDisplayName(self):
        return self.moduleName

    def getModuleDescription(self):
        return "Tjekker webhistorik-domæner mod Pi-hole DNS sinkhole"

    def getModuleVersionNumber(self):
        return "2.0"

    def isDataSourceIngestModuleFactory(self):
        return True

    def createDataSourceIngestModule(self, settings):
        return PiHoleLookupIngestModule()


class PiHoleLookupIngestModule(DataSourceIngestModule):

    def __init__(self):
        self.context = None
        self.moduleName = PiHoleLookupIngestModuleFactory.moduleName
        self.logger = IngestServices.getInstance().getLogger(self.moduleName)
        
        # Typer (Pre-loades i startup)
        self.type_url = None
        self.type_domain = None
        self.type_set_name = None
        self.type_comment = None
        self.type_assoc = None

        # *** KONFIGURATION ***
        self.PI_HOLE_DNS = "192.168.1.2" # <- CHANGE ME 
        self.domain_cache = {}

    def startUp(self, context):
        self.context = context
        self.logger.log(Level.INFO, "PiHoleLookup: StartUp. DNS: " + self.PI_HOLE_DNS)
        
        try:
            skCase = Case.getCurrentCase().getSleuthkitCase()
            self.type_url = skCase.getAttributeType("TSK_URL")
            self.type_domain = skCase.getAttributeType("TSK_DOMAIN")
            self.type_set_name = skCase.getAttributeType("TSK_SET_NAME")
            self.type_comment = skCase.getAttributeType("TSK_COMMENT")
            self.type_assoc = skCase.getAttributeType("TSK_ASSOCIATED_ARTIFACT")
        except Exception as e:
            self.logger.log(Level.SEVERE, "PiHoleLookup: Fejl ved init af attributter: " + str(e))

    def shutDown(self):
        self.logger.log(Level.INFO, "PiHoleLookup: ShutDown")

    def _is_ip(self, text):
        # Simpelt tjek om strengen starter med et tal (IP indikator)
        return re.match(r"^\d", text) is not None

    def _extract_domain(self, url):
        if not url: return None
        try:
            url = re.sub(r'^[a-zA-Z]+://', '', url)
            host = url.split('/')[0]
            host = host.split(':')[0] # Fjern port
            return host.strip()
        except:
            return None

    def _check_domain_with_pihole(self, domain):
        if domain in self.domain_cache:
            return self.domain_cache[domain]

        blocked = False
        try:
            cmd = ["nslookup", domain, self.PI_HOLE_DNS]
            proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            out, err = proc.communicate()
            text = str(out)

            # LOGIK V2.0:
            
            # 1. Hvis det er en IP (Reverse Lookup):
            #    Et succesfuldt svar har linjen "Name: ..."
            #    Et blokeret/fejlet svar har kun "Server: ..." og "Address: ..."
            if self._is_ip(domain):
                if "Name:" not in text:
                    blocked = True
            
            # 2. Hvis det er et Domæne (Forward Lookup):
            #    Kig efter NXDOMAIN eller 0.0.0.0
            else:
                if "NXDOMAIN" in text or "Non-existent domain" in text:
                    blocked = True
                elif "0.0.0.0" in text or "127.0.0.1" in text:
                    # Sikr at det ikke bare er server-headeren
                    if text.count("0.0.0.0") > 0 or text.count("127.0.0.1") > 0:
                        blocked = True

        except Exception as e:
            self.logger.log(Level.WARNING, "Lookup fejl: " + str(e))
            blocked = False

        self.domain_cache[domain] = blocked
        return blocked

    def process(self, dataSource, progressBar):
        self.logger.log(Level.INFO, "PiHoleLookup: Scanner " + dataSource.getName())
        
        case = Case.getCurrentCase()
        skCase = case.getSleuthkitCase()
        new_artifacts = ArrayList()

        try:
            art_type_id = BlackboardArtifact.ARTIFACT_TYPE.TSK_WEB_HISTORY.getTypeID()
            all_web_history = skCase.getBlackboardArtifacts(art_type_id)
        except:
            return IngestModule.ProcessResult.OK

        if not all_web_history:
            return IngestModule.ProcessResult.OK
            
        progressBar.switchToDeterminate(len(all_web_history))
        count = 0

        for art in all_web_history:
            if self.context.isJobCancelled():
                return IngestModule.ProcessResult.OK

            count += 1
            if count % 50 == 0:
                progressBar.progress(count)

            try:
                if art.getDataSource().getId() != dataSource.getId():
                    continue
            except:
                pass

            try:
                url_attr = art.getAttribute(self.type_url)
                domain_attr = art.getAttribute(self.type_domain)

                url_str = url_attr.getValueString() if url_attr else None
                
                # Prioriter at trække host direkte fra URL, da TSK_DOMAIN ofte er tom
                domain_str = self._extract_domain(url_str)
                
                # Hvis URL failer, prøv TSK_DOMAIN attributten
                if not domain_str and domain_attr:
                    domain_str = domain_attr.getValueString()

                if not domain_str:
                    continue

                # TJEK MOD PI-HOLE
                if self._check_domain_with_pihole(domain_str):
                    
                    # OPRET HIT
                    parent = art.getParent()
                    artifact = parent.newArtifact(BlackboardArtifact.ARTIFACT_TYPE.TSK_INTERESTING_ARTIFACT_HIT)
                    
                    artifact.addAttribute(BlackboardAttribute(
                        self.type_set_name, self.moduleName, "Pi-hole Blocked IOs"))
                        
                    artifact.addAttribute(BlackboardAttribute(
                        self.type_comment, self.moduleName, 
                        "Blokeret/Ingen DNS Record: " + str(domain_str)))
                    
                    artifact.addAttribute(BlackboardAttribute(
                        self.type_assoc, self.moduleName, art.getArtifactID()))
                    
                    if url_str:
                         artifact.addAttribute(BlackboardAttribute(
                            self.type_url, self.moduleName, url_str))
                    
                    if domain_str:
                         artifact.addAttribute(BlackboardAttribute(
                            self.type_domain, self.moduleName, domain_str))

                    new_artifacts.add(artifact)

            except Exception as e:
                self.logger.log(Level.WARNING, "Fejl i process loop: " + str(e))

        # FYR EVENTS AF
        if not new_artifacts.isEmpty():
            IngestServices.getInstance().fireModuleDataEvent(
                ModuleDataEvent(self.moduleName, BlackboardArtifact.ARTIFACT_TYPE.TSK_INTERESTING_ARTIFACT_HIT, new_artifacts)
            )
            self.logger.log(Level.INFO, "PiHoleLookup: Fandt " + str(new_artifacts.size()) + " hits.")

        return IngestModule.ProcessResult.OK