# -*- coding: utf-8 -*-
# MalwareIndicator.py
# Version 1.0 - Extensionless Task Support
# Fix: Fanger nu task-filer uden extension (f.eks. "services update").
# Formål: Dette er eksperimentelt script for at se indikatorer for malware på et system. Baseret på generisk malware opførsel.

import re
from java.util.logging import Level
from java.util import ArrayList
from java.lang import System 

from org.sleuthkit.autopsy.casemodule import Case
from org.sleuthkit.autopsy.ingest import IngestModuleFactoryAdapter
from org.sleuthkit.autopsy.ingest import DataSourceIngestModule
from org.sleuthkit.autopsy.ingest import IngestServices
from org.sleuthkit.autopsy.ingest import ModuleDataEvent
from org.sleuthkit.autopsy.ingest import IngestModule
from org.sleuthkit.datamodel import BlackboardArtifact
from org.sleuthkit.datamodel import BlackboardAttribute
from org.sleuthkit.datamodel import ReadContentInputStream

class MalwareIndicatorFactory(IngestModuleFactoryAdapter):
    moduleName = "Malware Indicator"
    def getModuleDisplayName(self): return self.moduleName
    def getModuleDescription(self): return "Scanner for malware og extensionless tasks."
    def getModuleVersionNumber(self): return "4.2"
    def isDataSourceIngestModuleFactory(self): return True
    def createDataSourceIngestModule(self, settings): return MalwareIndicatorModule()

class MalwareIndicatorModule(DataSourceIngestModule):

    def __init__(self):
        self.context = None
        self.moduleName = MalwareIndicatorFactory.moduleName
        self.logger = IngestServices.getInstance().getLogger(self.moduleName)
        
        self.SUSPICIOUS_NAMES = [
            "powershell", "pwsh", "cmd.exe", "conhost", "wscript", "cscript", 
            "sc.exe", "net.exe", "net1.exe", "whoami", "tasklist", "reg.exe", 
            "bitsadmin", "certutil", "rundll32", "regsvr32", "mshta", 
            "7zg.exe", "7z.exe", "rar.exe", "winrar", "tor.exe", "onion", 
            "mimikatz", "nmap", "wireshark", "metasploit", "cobaltstrike",
            "psexec", "paexec", "trickster", "trickbot", "emotet", "boda"
        ]

        self.SYSTEM_BINARIES = [
            "svchost.exe", "lsass.exe", "csrss.exe", "winlogon.exe", 
            "services.exe", "smss.exe", "explorer.exe", "taskhost.exe"
        ]

        self.PATH_WHITELIST = [
            "windows/system32", "windows/syswow64", "windows/winsxs",
            "program files/7-zip", "program files (x86)/7-zip"
        ]

        # Extensions til den generelle fil-scanning
        self.EXECUTABLE_EXTS = [".exe", ".bat", ".vbs", ".ps1", ".scr", ".js", ".hta"]

    def startUp(self, context):
        self.context = context
        self.logger.log(Level.INFO, "MalwareIndicator v4.2: StartUp")

    def to_unicode(self, text):
        if text is None: return u""
        try:
            s = str(text) 
            return s.decode('utf-8', 'ignore')
        except:
            try:
                return unicode(text, errors='ignore')
            except:
                return u"UNKNOWN_STRING"

    def is_whitelisted(self, path):
        if not path: return False
        p_lower = self.to_unicode(path).lower().replace(u"\\", u"/")
        for safe_path in self.PATH_WHITELIST:
            if safe_path in p_lower: return True
        return False

    def is_name_suspicious(self, filename):
        name_lower = self.to_unicode(filename).lower()
        for keyword in self.SUSPICIOUS_NAMES:
            if keyword in name_lower: return keyword
        return None

    def is_path_suspicious(self, path):
        if not path: return None
        p = self.to_unicode(path).lower().replace(u"\\", u"/")

        if u"start menu" in p and u"startup" in p: return "Persistence (Startup Folder)"
        if u"/users/public" in p: return "Suspicious Location (Public Folder)"
        if u"appdata" in p and u"/local/temp" in p: return "Suspicious Location (Temp)"
        if u"/downloads/" in p: return "Suspicious Location (Downloads)"

        if u"/programdata/" in p:
            try:
                parts = p.split(u"/programdata/")
                if len(parts) > 1 and parts[1].count(u"/") <= 1: return "Suspicious Location (ProgramData Shallow)"
            except: pass

        if u"/users/" in p and not u"/public" in p and not u"/default" in p:
            try:
                parts = p.split(u"/users/")
                if len(parts) > 1:
                    subpath = parts[1]
                    slash_count = subpath.count(u"/")
                    if subpath.endswith(u"/"): slash_count -= 1
                    if slash_count == 0: return "Suspicious Location (User Profile Root)"
            except: pass

        if u"/appdata/roaming/" in p:
            try:
                parts = p.split(u"/appdata/roaming/")
                if len(parts) > 1 and parts[1].count(u"/") <= 1: return "Suspicious Location (AppData Shallow)"
            except: pass
        
        return None

    def analyze_task_content(self, file):
        try:
            buffer = bytearray(4096)
            len_read = file.read(buffer, 0, 4096)
            # Læs som ren tekst (ignorer null bytes som kan forekomme i unicode filer)
            content = "".join([chr(b) for b in buffer if 32 <= b <= 126])
            
            # Led efter .exe
            match = re.search(r"([a-zA-Z]:\\[^\<\>\"\|]+\.exe)", content, re.IGNORECASE)
            if match:
                return " -> Trigger: " + match.group(1)
            
            if ".exe" in content.lower():
                return " -> Suspicious content (EXE reference found)"
        except: pass
        return ""

    def process(self, dataSource, progressBar):
        self.logger.log(Level.INFO, "MalwareIndicator: Scanner...")
        case = Case.getCurrentCase()
        skCase = case.getSleuthkitCase()
        new_artifacts = ArrayList()

        # --- FASE 1: PREFETCH SCAN (Uændret) ---
        try:
            files_pf = skCase.findAllFilesWhere("name LIKE '%.pf'")
            for f in files_pf:
                if self.context.isJobCancelled(): return IngestModule.ProcessResult.OK
                try:
                    fname = f.getName()
                    keyword = self.is_name_suspicious(fname)
                    if keyword:
                        self.create_artifact(f, "Malware Execution (Prefetch)", 
                                             "Kørsel bekræftet: " + self.to_unicode(fname), new_artifacts)
                except: pass
        except: pass

        # --- FASE 2: EXECUTABLE & SYSTEM SCAN (Uændret) ---
        ext_clause = " OR ".join(["name LIKE '%" + ext + "'" for ext in self.EXECUTABLE_EXTS])
        
        try:
            files_exe = skCase.findAllFilesWhere("(" + ext_clause + ")")
            self.logger.log(Level.INFO, "MalwareIndicator: Fandt " + str(len(files_exe)) + " filer at scanne.")
        except: pass

        count = 0
        if files_exe:
            progressBar.switchToDeterminate(len(files_exe))
        
        for f in files_exe:
            count += 1
            if count % 500 == 0: progressBar.progress(count)
            if self.context.isJobCancelled(): return IngestModule.ProcessResult.OK
            
            try:
                if f.getDataSource().getId() != dataSource.getId(): continue
                path = f.getUniquePath()
                filename = f.getName()
                filename_u = self.to_unicode(filename)
                filename_lower = filename_u.lower()
                
                if self.is_whitelisted(path): continue

                is_hit = False
                category = ""
                description = ""

                # Generel Heuristik (som før)
                bad_path_desc = self.is_path_suspicious(path)
                
                if bad_path_desc:
                     is_hit = True
                     category = "Suspicious File Location"
                     description = bad_path_desc + u": " + filename_u

                if not is_hit:
                    if (filename_lower.endswith(u".exe") or filename_lower.endswith(u".scr")):
                        if any(x in filename_lower for x in [u".pdf.", u".doc.", u".xls.", u".jpg."]):
                            is_hit = True
                            category = "Heuristics (Double Extension)"
                            description = u"Potentielt ondsindet fil: " + filename_u
                    elif filename_lower in self.SYSTEM_BINARIES:
                        is_hit = True
                        category = "Heuristics (Masquerading)"
                        description = u"Kritisk: Systemfil uden for System32: " + self.to_unicode(path)
                    else:
                        bad_name = self.is_name_suspicious(filename)
                        if bad_name:
                            is_hit = True
                            category = "Suspicious File Name"
                            description = u"Mistænkeligt navn: " + filename_u

                if is_hit:
                    self.create_artifact(f, category, description, new_artifacts)
            except: pass

        # --- FASE 3: TARGETED TASK SCAN (NYT I V4.2) ---
        # Vi scanner ALT i Tasks mapperne, uanset extension!
        self.logger.log(Level.INFO, "MalwareIndicator: Scanner Tasks mapper...")
        try:
            # SQL Query for at finde alt i Tasks folderne
            tasks = skCase.findAllFilesWhere("parent_path LIKE '%/system32/tasks/%' OR parent_path LIKE '%/windows/tasks/%'")
            
            for f in tasks:
                if self.context.isJobCancelled(): return IngestModule.ProcessResult.OK
                if f.isDir(): continue # Spring mapper over
                
                path = f.getUniquePath()
                p_lower = self.to_unicode(path).lower().replace(u"\\", u"/")
                filename = f.getName()

                # Tjek 1: Legacy Tasks (.job i Windows/Tasks)
                if u"/windows/tasks/" in p_lower:
                     # Hvis den ligger direkte i Tasks roden og ikke er GoogleUpdate
                     if u"/" not in p_lower.split(u"/windows/tasks/")[1].strip(u"/"):
                         if not u"googleupdate" in p_lower and not u"sahagent" in p_lower:
                            self.create_artifact(f, "Persistence (Legacy Task File)", "Gammeldags Task fil fundet", new_artifacts)

                # Tjek 2: Modern Tasks (XML i System32/Tasks)
                if u"/system32/tasks/" in p_lower:
                    try:
                        relative = p_lower.split(u"/system32/tasks/")[1]
                        # Hvis filen ligger direkte i roden (SHALLOW)
                        if u"/" not in relative.strip(u"/"):
                            # Her fanger vi 'services update' selvom den ikke har .xml extension!
                            content_info = self.analyze_task_content(f)
                            self.create_artifact(f, "CRITICAL: Persistence (Suspicious Task Location)", 
                                                 u"Task fundet i rod. " + content_info, new_artifacts)
                    except: pass
        except Exception as e:
             self.logger.log(Level.WARNING, "Fejl ved Task scan: " + str(e))

        # Publicer artifacts
        if not new_artifacts.isEmpty():
            try:
                IngestServices.getInstance().fireModuleDataEvent(
                    ModuleDataEvent(self.moduleName, BlackboardArtifact.ARTIFACT_TYPE.TSK_INTERESTING_FILE_HIT, new_artifacts)
                )
            except: pass

        return IngestModule.ProcessResult.OK

    def create_artifact(self, abstract_file, category, comment, list_obj):
        try:
            art = abstract_file.newArtifact(BlackboardArtifact.ARTIFACT_TYPE.TSK_INTERESTING_FILE_HIT)
            art.addAttribute(BlackboardAttribute(BlackboardAttribute.ATTRIBUTE_TYPE.TSK_SET_NAME, self.moduleName, category))
            art.addAttribute(BlackboardAttribute(BlackboardAttribute.ATTRIBUTE_TYPE.TSK_COMMENT, self.moduleName, comment))
            list_obj.add(art)
        except: pass